/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx__valgrind_h_
#define _mx__valgrind_h_

#include "mx_auto_config.h"
#include "mx_io.h"
#include "mx_io_impl.h"
#include "mx_debug.h"

#if MX_VALGRIND_DEBUG && !defined MX_KERNEL

/*
 * Valgrind support to check memory access and allocation.
 * Use "valgrind --weird-hacks=lax-ioctls myprogram" to check your program
 * (and the MX lib). 
 * The option name has changed in more recent valgrind versions, use:
 *      "valgrind --sim-hints=lax-ioctls myprogram"
 */

#include "valgrind/memcheck.h"

/* Mark a memory buffer as non-accessible, accessible or accessible+initialized */
#define MX_VALGRIND_MEMORY_MAKE_NOACCESS(p, s) VALGRIND_MAKE_NOACCESS(p, s)
#define MX_VALGRIND_MEMORY_MAKE_WRITABLE(p, s) VALGRIND_MAKE_WRITABLE(p, s)
#define MX_VALGRIND_MEMORY_MAKE_READABLE(p, s) VALGRIND_MAKE_READABLE(p, s)

/* Mark segments as filled with received data, up to length */
static inline void
MX_VALGRIND_MEMORY_MAKE_SEGMENTS_READABLE(mx_segment_t * segments,
					  uint32_t segment_count,
					  uint32_t length)
{
  while (length > 0) {
    uint32_t chunk = length > segments[0].segment_length ? segments[0].segment_length : length;
    VALGRIND_MAKE_READABLE(segments[0].segment_ptr, chunk);
    length -= chunk;
    segments++;
  }  
}

/* Check that input buffers are readable and output buffers are writable */
static inline void
MX_VALGRIND_PRE_IOCTL_CHECK(int cmd, void *buffer)
{
  switch (cmd) {

  /* IOCTL without any argument */
  case MX_TEST_IOCTL:
  case MX_SET_RAW:
  case MX_SET_ROUTE_BEGIN:
  case MX_SET_ROUTE_END:
  case MX_WAKE:
  case MX_APP_WAKE:
    break;

  /* IOCTL with an input uint32_t */
  case MX_DEREGISTER:
  case MX_ARM_TIMER:
    VALGRIND_CHECK_READABLE(buffer, sizeof(uint32_t));
    break;

  /* IOCTL with an output uint32_t */
  case MX_GET_INSTANCE_COUNT:
  case MX_GET_MAX_INSTANCE:
  case MX_GET_MAX_SEND_HANDLES:
  case MX_GET_MAX_ENDPOINTS:
  case MX_GET_MAX_PEERS:
  case MX_GET_SMALL_MESSAGE_THRESHOLD:
  case MX_GET_MEDIUM_MESSAGE_THRESHOLD:
  case MX_GET_MAX_RDMA_WINDOWS:
  case MX_GET_BOARD_STATUS:
    VALGRIND_CHECK_WRITABLE(buffer, sizeof(uint32_t));
    break;

  /* IOCTL with an input/output uint32_t */
  case MX_GET_NUM_PORTS:
  case MX_GET_SERIAL_NUMBER:
    VALGRIND_CHECK_READABLE(buffer, sizeof(uint32_t));
    VALGRIND_CHECK_WRITABLE(buffer, sizeof(uint32_t));
    break;

  case MX_SET_ENDPOINT:
  {
    mx_set_endpt_t * x = (mx_set_endpt_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->endpoint);
    VALGRIND_CHECK_WRITABLE(&x->session_id, sizeof(x->session_id));
    break;
  }

  case MX_WAIT:
  case MX_APP_WAIT:
  {
    mx_wait_t * x = (mx_wait_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->timeout);
    VALGRIND_CHECK_WRITABLE(&x->mcp_wake_events, sizeof(x->mcp_wake_events));
    VALGRIND_CHECK_WRITABLE(&x->status, sizeof(x->status));
    break;
  }

  case MX_GET_COPYBLOCKS:
  {
    mx_get_copyblock_t * x = (mx_get_copyblock_t *) buffer;
    VALGRIND_CHECK_WRITABLE(&x->sendq_offset, sizeof(x->sendq_offset));
    VALGRIND_CHECK_WRITABLE(&x->sendq_len, sizeof(x->sendq_len));
    VALGRIND_CHECK_WRITABLE(&x->recvq_offset, sizeof(x->recvq_offset));
    VALGRIND_CHECK_WRITABLE(&x->recvq_len, sizeof(x->recvq_len));
    VALGRIND_CHECK_WRITABLE(&x->eventq_offset, sizeof(x->eventq_offset));
    VALGRIND_CHECK_WRITABLE(&x->eventq_len, sizeof(x->eventq_len));
    VALGRIND_CHECK_WRITABLE(&x->user_mmapped_sram_offset, sizeof(x->user_mmapped_sram_offset));
    VALGRIND_CHECK_WRITABLE(&x->user_mmapped_sram_len, sizeof(x->user_mmapped_sram_len));
    VALGRIND_CHECK_WRITABLE(&x->user_mmapped_zreq_offset, sizeof(x->user_mmapped_zreq_offset));
    VALGRIND_CHECK_WRITABLE(&x->user_mmapped_zreq_len, sizeof(x->user_mmapped_zreq_len));
    VALGRIND_CHECK_WRITABLE(&x->user_reqq_offset, sizeof(x->user_reqq_offset));
    VALGRIND_CHECK_WRITABLE(&x->user_reqq_len, sizeof(x->user_reqq_len));
    VALGRIND_CHECK_WRITABLE(&x->user_dataq_offset, sizeof(x->user_dataq_offset));
    VALGRIND_CHECK_WRITABLE(&x->user_dataq_len, sizeof(x->user_dataq_len));
    VALGRIND_CHECK_WRITABLE(&x->kernel_window_offset, sizeof(x->kernel_window_offset));
    VALGRIND_CHECK_WRITABLE(&x->kernel_window_len, sizeof(x->kernel_window_len));
    break;
  }

  case MX_GET_NIC_ID:
  {
    mx_get_nic_id_t * x = (mx_get_nic_id_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->board_number);
    VALGRIND_CHECK_WRITABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_REGISTER:
  {
    mx_reg_t * x = (mx_reg_t *) buffer;
    uint32_t nsegs = x->nsegs;
    VALGRIND_CHECK_DEFINED(x->rdma_id);
    VALGRIND_CHECK_DEFINED(x->memory_context);
    VALGRIND_CHECK_DEFINED(nsegs);
    VALGRIND_CHECK_DEFINED(x->segs.vaddr);
    if (nsegs == 1) {
      VALGRIND_CHECK_DEFINED(x->segs.len);
    } else {
      mx_reg_seg_t * segs = (mx_reg_seg_t *)(uintptr_t) x->segs.vaddr;
      int i;
      for(i=0; i<nsegs; i++) {
	VALGRIND_CHECK_DEFINED(segs[i].vaddr);
	VALGRIND_CHECK_DEFINED(segs[i].len);
      }
    }
    break;
  }

  case MX_NIC_ID_TO_PEER_INDEX:
  {
    mx_lookup_peer_t * x = (mx_lookup_peer_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->nic_id);
    VALGRIND_CHECK_DEFINED(x->board_number);
    VALGRIND_CHECK_WRITABLE(&x->index, sizeof(x->index));
    break;
  }

  case MX_PEER_INDEX_TO_NIC_ID:
  {
    mx_lookup_peer_t * x = (mx_lookup_peer_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->index);
    VALGRIND_CHECK_WRITABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_NIC_ID_TO_HOSTNAME:
  {
    mx_nic_id_hostname_t * x = (mx_nic_id_hostname_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->nic_id);
    VALGRIND_CHECK_DEFINED(x->va);
    VALGRIND_CHECK_DEFINED(x->len);
    VALGRIND_CHECK_WRITABLE(x->va, x->len);
    break;
  }

  case MX_HOSTNAME_TO_NIC_ID:
  {
    mx_nic_id_hostname_t * x = (mx_nic_id_hostname_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->va);
    VALGRIND_CHECK_DEFINED(x->len);
    VALGRIND_CHECK_READABLE(x->va, x->len);
    VALGRIND_CHECK_WRITABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_WAKE_ENDPOINT:
  {
    mx_wake_endpt_t * x = (mx_wake_endpt_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->endpt);
    break;
  }

  case MX_GET_VERSION:
  {
    mx_get_version_t * x = (mx_get_version_t *) buffer;
    VALGRIND_CHECK_WRITABLE(x, sizeof(*x));
    break;
  }

  case MX_DIRECT_GETV:
  {
    mx_direct_getv_t * x = (mx_direct_getv_t *) buffer;
    uint32_t src_nsegs = x->src_nsegs;
    uint32_t dst_nsegs = x->dst_nsegs;
    VALGRIND_CHECK_DEFINED(x->length);
    VALGRIND_CHECK_DEFINED(x->src_board_num);
    VALGRIND_CHECK_DEFINED(x->src_endpt);
    VALGRIND_CHECK_DEFINED(x->src_session);
    VALGRIND_CHECK_DEFINED(src_nsegs);
    VALGRIND_CHECK_DEFINED(x->src_segs.vaddr);
    if (src_nsegs == 1) {
      VALGRIND_CHECK_DEFINED(x->src_segs.len);
    } else {
      /* no way to check src since it's in another address space */
    }
    VALGRIND_CHECK_DEFINED(dst_nsegs);
    VALGRIND_CHECK_DEFINED(x->dst_segs.vaddr);
    if (dst_nsegs == 1) {
      VALGRIND_CHECK_DEFINED(x->dst_segs.len);
    } else {
      mx_shm_seg_t * dst_segs = (mx_shm_seg_t *)(uintptr_t) x->dst_segs.vaddr;
      int i;
      for(i=0; i<dst_nsegs; i++) {
	VALGRIND_CHECK_DEFINED(dst_segs[i].vaddr);
	VALGRIND_CHECK_DEFINED(dst_segs[i].len);
      }
    }
    break;
  }

  case MX_GET_BOARD_TYPE:
  {
    mx_get_board_val_t * x = (mx_get_board_val_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->board_number);
    VALGRIND_CHECK_WRITABLE(&x->val, sizeof(x->val));
    break;
  }

  case MX_SET_ROUTE:
  {
    mx_set_route_t * x = (mx_set_route_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->route_pointer);
    VALGRIND_CHECK_READABLE(x->route_pointer, x->route_length);
    VALGRIND_CHECK_DEFINED(x->mac_low32);
    VALGRIND_CHECK_DEFINED(x->mac_high16);
    VALGRIND_CHECK_DEFINED(x->source_port);
    VALGRIND_CHECK_DEFINED(x->route_length);
    VALGRIND_CHECK_DEFINED(x->host_type);
    break;
  }

  case MX_RAW_SEND:
  {
    mx_raw_send_t * x = (mx_raw_send_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->data_pointer);
    VALGRIND_CHECK_DEFINED(x->route_pointer);
    VALGRIND_CHECK_DEFINED(x->context);
    VALGRIND_CHECK_DEFINED(x->route_length);
    VALGRIND_CHECK_DEFINED(x->physical_port);
    VALGRIND_CHECK_DEFINED(x->buffer_length);
    break;
  }

  case MX_RAW_GET_NEXT_EVENT:
  {
    mx_raw_next_event_t * x = (mx_raw_next_event_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->recv_buffer);
    VALGRIND_CHECK_DEFINED(x->recv_bytes);
    VALGRIND_CHECK_DEFINED(x->timeout);
    VALGRIND_CHECK_WRITABLE(&x->status, sizeof(x->status));
    /* TODO */
    break;
  }

#if MX_DRIVER_API_MAGIC >= 0x500
  case MX_RAW_SET_NIC_REPLY_INFO:
  {
    mx_raw_set_nic_reply_info_t * x = (mx_raw_set_nic_reply_info_t *) buffer;

    VALGRIND_CHECK_DEFINED(x->len);
    VALGRIND_CHECK_READABLE(x->blob_va, x->len);
    break;
  }
#endif

  case MX_SET_MAPPER_STATE:
  {
    mx_mapper_state_t * x = (mx_mapper_state_t *) buffer;
    VALGRIND_CHECK_DEFINED(x->board_number);
    VALGRIND_CHECK_DEFINED(x->mapper_mac);
    VALGRIND_CHECK_DEFINED(x->iport);
    VALGRIND_CHECK_DEFINED(x->map_version);
    VALGRIND_CHECK_DEFINED(x->num_hosts);
    VALGRIND_CHECK_DEFINED(x->network_configured);
    VALGRIND_CHECK_DEFINED(x->routes_valid);
    VALGRIND_CHECK_DEFINED(x->level);
    VALGRIND_CHECK_DEFINED(x->flags);
    break;
  }

  case MX_GET_PRODUCT_CODE:
  {
    mx_get_eeprom_string_t * x = (mx_get_eeprom_string_t *) buffer;
    char * ptr = (char *)((uintptr_t) (x->buffer));
    VALGRIND_CHECK_DEFINED(x->board_number);
    VALGRIND_CHECK_DEFINED(x->buffer);
    VALGRIND_CHECK_WRITABLE(ptr, MX_MAX_STR_LEN);
    break;
  }

  case MX_NIC_ID_TO_BOARD_NUM:
    {
      mx_nic_id_to_board_num_t *x = (mx_nic_id_to_board_num_t *)buffer;
      VALGRIND_CHECK_DEFINED(x->nic_id);
      VALGRIND_CHECK_WRITABLE(&x->board_number, sizeof(x->board_number));
      break;
    }

  /* TODO */
  case MX_CRASHDUMP:
  case MX_GET_COUNTERS:
  case MX_CLEAR_COUNTERS:
  case MX_PIN_SEND:
  case MX_GET_COUNTERS_STRINGS:
  case MX_GET_LOGGING:
  case MX_GET_LOGGING_STRINGS:
  case MX_GET_MAPPER_MSGBUF_SIZE:
  case MX_GET_MAPPER_MAPBUF_SIZE:
  case MX_GET_MAPPER_MSGBUF:
  case MX_GET_MAPPER_MAPBUF:
  case MX_GET_PEER_FORMAT:
  case MX_GET_ROUTE_SIZE:
  case MX_GET_PEER_TABLE:
  case MX_GET_ROUTE_TABLE:
  case MX_PAUSE_MAPPER:
  case MX_RESUME_MAPPER:
  case MX_GET_OPENER:
  case MX_MMAP:
  case MX_MUNMAP:
  case MX_CLEAR_PEER_NAMES:
  case MX_SET_HOSTNAME:
  case MX_GET_CACHELINE_SIZE:
  case MX_GET_LINK_STATE:
  case MX_PIN_RECV:
  case MX_GET_IRQ_COUNTERS:
  case MX_RUN_DMABENCH:
  case MX_CLEAR_WAIT:
  case MX_RAW_GET_PARAMS:
  case MX_RAW_CLEAR_ROUTES:
  case MX_GET_CPU_FREQ:
  case MX_GET_PCI_FREQ:
  case MX_GET_MAPPER_STATE:
  case MX_GET_INTR_COAL:
  case MX_SET_INTR_COAL:
  case MX_RECOVER_ENDPOINT:
  case MX_REMOVE_PEER:
  case MX_DIRECT_GET:
  case MX_GET_PART_NUMBER:
  case MX_GET_SRAM_SIZE:
  case MX_GET_DUMP_REG_COUNT:
  case MX_WRITE_PIO_REQ:
#if MX_DRIVER_API_MAGIC >= 0x500
  case MX_WAIT_FOR_RECOVERY:
  case MX_GET_SOME_COUNTERS:
#endif
    VALGRIND_PRINTF("Unhandled PRE IOCTL %x\n", cmd);
    break;
  default:
    mx_always_assert(0);
  }
}

/* Mark that output buffers as readable */
static inline void
MX_VALGRIND_POST_IOCTL_CHECK(int cmd, void *buffer)
{
  switch (cmd) {

  /* IOCTL without any argument */
  case MX_TEST_IOCTL:
  case MX_SET_RAW:
  case MX_SET_ROUTE_BEGIN:
  case MX_SET_ROUTE_END:
  case MX_WAKE:
  case MX_APP_WAKE:
    break;

  /* IOCTL with an input uint32_t */
  case MX_DEREGISTER:
  case MX_ARM_TIMER:
    break;

  /* IOCTL with an output uint32_t */
  case MX_GET_INSTANCE_COUNT:
  case MX_GET_MAX_INSTANCE:
  case MX_GET_MAX_SEND_HANDLES:
  case MX_GET_MAX_ENDPOINTS:
  case MX_GET_MAX_PEERS:
  case MX_GET_SMALL_MESSAGE_THRESHOLD:
  case MX_GET_MEDIUM_MESSAGE_THRESHOLD:
  case MX_GET_MAX_RDMA_WINDOWS:
  case MX_GET_BOARD_STATUS:
    VALGRIND_MAKE_READABLE(buffer, sizeof(uint32_t));
    break;

  /* IOCTL with an input/output uint32_t */
  case MX_GET_NUM_PORTS:
  case MX_GET_SERIAL_NUMBER:
    VALGRIND_MAKE_READABLE(buffer, sizeof(uint32_t));
    break;

  case MX_SET_ENDPOINT:
  {
    mx_set_endpt_t * x = (mx_set_endpt_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->session_id, sizeof(x->session_id));
    break;
  }

  case MX_WAIT:
  case MX_APP_WAIT:
  {
    mx_wait_t * x = (mx_wait_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->mcp_wake_events, sizeof(x->mcp_wake_events));
    VALGRIND_MAKE_READABLE(&x->status, sizeof(x->status));
    break;
  }

  case MX_GET_COPYBLOCKS:
  {
    mx_get_copyblock_t * x = (mx_get_copyblock_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->sendq_offset, sizeof(x->sendq_offset));
    VALGRIND_MAKE_READABLE(&x->sendq_len, sizeof(x->sendq_len));
    VALGRIND_MAKE_READABLE(&x->recvq_offset, sizeof(x->recvq_offset));
    VALGRIND_MAKE_READABLE(&x->recvq_len, sizeof(x->recvq_len));
    VALGRIND_MAKE_READABLE(&x->eventq_offset, sizeof(x->eventq_offset));
    VALGRIND_MAKE_READABLE(&x->eventq_len, sizeof(x->eventq_len));
    VALGRIND_MAKE_READABLE(&x->user_mmapped_sram_offset, sizeof(x->user_mmapped_sram_offset));
    VALGRIND_MAKE_READABLE(&x->user_mmapped_sram_len, sizeof(x->user_mmapped_sram_len));
    VALGRIND_MAKE_READABLE(&x->user_mmapped_zreq_offset, sizeof(x->user_mmapped_zreq_offset));
    VALGRIND_MAKE_READABLE(&x->user_mmapped_zreq_len, sizeof(x->user_mmapped_zreq_len));
    VALGRIND_MAKE_READABLE(&x->user_reqq_offset, sizeof(x->user_reqq_offset));
    VALGRIND_MAKE_READABLE(&x->user_reqq_len, sizeof(x->user_reqq_len));
    VALGRIND_MAKE_READABLE(&x->user_dataq_offset, sizeof(x->user_dataq_offset));
    VALGRIND_MAKE_READABLE(&x->user_dataq_len, sizeof(x->user_dataq_len));
    VALGRIND_MAKE_READABLE(&x->kernel_window_offset, sizeof(x->kernel_window_offset));
    VALGRIND_MAKE_READABLE(&x->kernel_window_len, sizeof(x->kernel_window_len));
    break;
  }

  case MX_GET_NIC_ID:
  {
    mx_get_nic_id_t * x = (mx_get_nic_id_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_REGISTER:
    break;

  case MX_NIC_ID_TO_PEER_INDEX:
  {
    mx_lookup_peer_t * x = (mx_lookup_peer_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->index, sizeof(x->index));
    break;
  }

  case MX_PEER_INDEX_TO_NIC_ID:
  {
    mx_lookup_peer_t * x = (mx_lookup_peer_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_NIC_ID_TO_HOSTNAME:
  {
    mx_nic_id_hostname_t * x = (mx_nic_id_hostname_t *) buffer;
    VALGRIND_MAKE_READABLE(x->va, x->len);
    break;
  }

  case MX_HOSTNAME_TO_NIC_ID:
  {
    mx_nic_id_hostname_t * x = (mx_nic_id_hostname_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->nic_id, sizeof(x->nic_id));
    break;
  }

  case MX_WAKE_ENDPOINT:
    break;

  case MX_GET_VERSION:
  {
    mx_get_version_t * x = (mx_get_version_t *) buffer;
    VALGRIND_MAKE_READABLE(x, sizeof(*x));
    break;
  }

  case MX_DIRECT_GETV:
    break;

  case MX_GET_BOARD_TYPE:
  {
    mx_get_board_val_t * x = (mx_get_board_val_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->val, sizeof(x->val));
    break;
  }

  case MX_SET_ROUTE:
    break;

  case MX_RAW_SEND:
    break;

  case MX_RAW_GET_NEXT_EVENT:
  {
    mx_raw_next_event_t * x = (mx_raw_next_event_t *) buffer;
    VALGRIND_MAKE_READABLE(&x->status, sizeof(x->status));
    if (x->status == MX_KRAW_RECV_COMPLETE) {
      VALGRIND_MAKE_READABLE(&x->incoming_port, sizeof(x->incoming_port));
      VALGRIND_MAKE_READABLE(&x->recv_bytes, sizeof(x->recv_bytes));
      VALGRIND_MAKE_READABLE(&x->recv_buffer, sizeof(x->recv_buffer));
      VALGRIND_MAKE_READABLE(x->recv_buffer, x->recv_bytes);
    } else if (x->status == MX_KRAW_SEND_COMPLETE) {
      VALGRIND_MAKE_READABLE(&x->context, sizeof(x->context));
    }
    break;
  }

  case MX_SET_MAPPER_STATE:
#if MX_DRIVER_API_MAGIC >= 0x500
  case MX_RAW_SET_NIC_REPLY_INFO:
#endif
    break;

  case MX_GET_PRODUCT_CODE:
  {
    mx_get_eeprom_string_t * x = (mx_get_eeprom_string_t *) buffer;
    char * ptr = (char *)((uintptr_t) (x->buffer));
    VALGRIND_MAKE_READABLE(ptr, MX_MAX_STR_LEN);
    break;
  }

  case MX_NIC_ID_TO_BOARD_NUM:
    {
      mx_nic_id_to_board_num_t *x = (mx_nic_id_to_board_num_t *)buffer;
      VALGRIND_MAKE_READABLE(&x->board_number, sizeof(x->board_number));
      break;
    }

  /* TODO */
  case MX_CRASHDUMP:
  case MX_GET_COUNTERS:
  case MX_CLEAR_COUNTERS:
  case MX_PIN_SEND:
  case MX_GET_COUNTERS_STRINGS:
  case MX_GET_LOGGING:
  case MX_GET_LOGGING_STRINGS:
  case MX_GET_MAPPER_MSGBUF_SIZE:
  case MX_GET_MAPPER_MAPBUF_SIZE:
  case MX_GET_MAPPER_MSGBUF:
  case MX_GET_MAPPER_MAPBUF:
  case MX_GET_PEER_FORMAT:
  case MX_GET_ROUTE_SIZE:
  case MX_GET_PEER_TABLE:
  case MX_GET_ROUTE_TABLE:
  case MX_PAUSE_MAPPER:
  case MX_RESUME_MAPPER:
  case MX_GET_OPENER:
  case MX_MMAP:
  case MX_MUNMAP:
  case MX_CLEAR_PEER_NAMES:
  case MX_SET_HOSTNAME:
  case MX_GET_CACHELINE_SIZE:
  case MX_GET_LINK_STATE:
  case MX_PIN_RECV:
  case MX_GET_IRQ_COUNTERS:
  case MX_RUN_DMABENCH:
  case MX_CLEAR_WAIT:
  case MX_RAW_GET_PARAMS:
  case MX_RAW_CLEAR_ROUTES:
  case MX_GET_CPU_FREQ:
  case MX_GET_PCI_FREQ:
  case MX_GET_MAPPER_STATE:
  case MX_GET_INTR_COAL:
  case MX_SET_INTR_COAL:
  case MX_RECOVER_ENDPOINT:
  case MX_REMOVE_PEER:
  case MX_DIRECT_GET:
  case MX_GET_PART_NUMBER:
  case MX_GET_SRAM_SIZE:
  case MX_GET_DUMP_REG_COUNT:
  case MX_WRITE_PIO_REQ:
#if MX_DRIVER_API_MAGIC >= 0x500
  case MX_WAIT_FOR_RECOVERY:
  case MX_GET_SOME_COUNTERS:
#endif
    VALGRIND_PRINTF("Unhandled POST IOCTL %x\n", cmd);
    break;
  default:
    mx_always_assert(0);
  }
}

#else /* MX_VALGRIND_DEBUG && !defined MX_KERNEL */

#define MX_VALGRIND_MEMORY_MAKE_NOACCESS(p, s)
#define MX_VALGRIND_MEMORY_MAKE_WRITABLE(p, s)
#define MX_VALGRIND_MEMORY_MAKE_READABLE(p, s)
#define MX_VALGRIND_MEMORY_MAKE_SEGMENTS_READABLE(s, c, l)
#define MX_VALGRIND_PRE_IOCTL_CHECK(c, b)
#define MX_VALGRIND_POST_IOCTL_CHECK(c, b)

#endif /* !MX_VALGRIND_DEBUG || defined MX_KERNEL */

#endif /* _mx__valgrind_h_ */
